<?php
// Funkcja do skanowania folderów i plików
function scanDirectory($dir) {
    $files = [];
    if (is_dir($dir)) {
        // Użyj DirectoryIterator dla potencjalnie lepszej obsługi błędów i wydajności
        try {
            $iterator = new DirectoryIterator($dir);
            foreach ($iterator as $fileinfo) {
                if ($fileinfo->isDot()) {
                    continue; // Pomiń '.' i '..'
                }

                $itemName = $fileinfo->getFilename();
                $path = $fileinfo->getPathname();

                if ($fileinfo->isDir()) {
                    $files[] = [
                        'name' => $itemName,
                        'type' => 'directory',
                        'path' => $path,
                        'children' => scanDirectory($path) // Rekurencyjne skanowanie podkatalogów
                    ];
                } elseif ($fileinfo->isFile() && strtolower($fileinfo->getExtension()) === 'glb') { // Interesują nas tylko pliki .glb (małe/wielkie litery)
                    $files[] = [
                        'name' => $itemName,
                        'type' => 'file',
                        'path' => $path
                    ];
                }
            }
        } catch (Exception $e) {
            // Obsłuż błąd, np. brak uprawnień do odczytu katalogu
            error_log("Error scanning directory '$dir': " . $e->getMessage());
            // Można by zwrócić pustą tablicę lub specjalny obiekt błędu
            return [];
        }
    }
    // Sortowanie: najpierw katalogi, potem pliki, alfabetycznie w każdej grupie
    usort($files, function($a, $b) {
        if ($a['type'] === 'directory' && $b['type'] === 'file') {
            return -1;
        } elseif ($a['type'] === 'file' && $b['type'] === 'directory') {
            return 1;
        } else {
            // Użyj sortowania naturalnego dla lepszego porządku (np. plik10 po plik2)
            return strnatcasecmp($a['name'], $b['name']);
        }
    });
    return $files;
}

// Główny katalog z modelami 3D
$rootDir = 'pliki3d'; // Upewnij się, że ten katalog istnieje i zawiera pliki/podkatalogi
$filesTree = scanDirectory($rootDir);

// Funkcja do renderowania drzewa katalogów jako HTML
function renderTree($items, $level = 0) {
    $html = '';
    foreach ($items as $item) {
        $padding = $level * 15; // Wcięcie zależne od poziomu zagnieżdżenia
        // Użyj htmlspecialchars dla bezpieczeństwa przed XSS
        $safeName = htmlspecialchars($item['name'], ENT_QUOTES, 'UTF-8');
        $safePath = htmlspecialchars($item['path'], ENT_QUOTES, 'UTF-8');

        if ($item['type'] === 'directory') {
            $html .= '<div class="tree-item" style="padding-left: ' . $padding . 'px">';
            // Klikalny element katalogu z ikoną i przełącznikiem (+/-)
            $html .= '<div class="directory" onclick="toggleDirectory(this)" title="' . $safeName . '">';
            $html .= '<span class="toggle">+</span> 📁 ' . $safeName;
            $html .= '</div>';
            // Kontener na dzieci (pliki/podkatalogi), początkowo ukryty
            $html .= '<div class="tree-children" style="display: none">';
            // Rekurencyjne renderowanie zawartości katalogu (tylko jeśli są dzieci)
            if (!empty($item['children'])) {
                $html .= renderTree($item['children'], $level + 1);
            }
            $html .= '</div>';
            $html .= '</div>';
        } else { // Jeśli to plik .glb
            $html .= '<div class="tree-item" style="padding-left: ' . $padding . 'px">';
            // Klikalny element pliku z ikoną i ścieżką w atrybucie data-path
            $html .= '<div class="file" onclick="loadModel(\'' . $safePath . '\', this)" data-path="' . $safePath . '" title="' . $safeName . '">';
            $html .= '❒  ' . $safeName;
            $html .= '</div>';
            $html .= '</div>';
        }
    }
    return $html;
}

// Pobierz aktualny URL strony (bez parametrów i fragmentu)
$scheme = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http";
$host = $_SERVER['HTTP_HOST'];
$uri = strtok($_SERVER["REQUEST_URI"], '?'); // Usuń query string
$currentUrlBase = $scheme . "://" . $host . $uri;

// Początkowy URL do udostępnienia/kopiowania (może zostać zaktualizowany przez JS)
$initialShareUrl = $currentUrlBase;

?>
<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Galeria 3D - Zmienny Widok Siatki</title>
    <script src="https://cdn.jsdelivr.net/npm/three@0.132.2/build/three.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/three@0.132.2/examples/js/loaders/GLTFLoader.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/three@0.132.2/examples/js/controls/OrbitControls.js"></script>
    <style>
        /* === Style CSS podstawowe (bez zmian) === */
        :root {
             /* Definicje kolorów dla łatwiejszej zmiany */
            --bg-main: #333;
            --bg-header: #222;
            --bg-explorer: #444;
            --bg-copy-panel: #555;
            --bg-viewer: #222;
            --bg-directory: #555;
            --bg-directory-hover: #666;
            --bg-file: #3a3a3a;
            --bg-file-hover: #4a4a4a;
            --bg-file-active: #5a5a5a;
            --text-light: #fff;
            --text-dark: #333;
            --accent-color: #7fff7f; /* Zielony neon */
            --border-color: #555;
            --border-accent: var(--accent-color);
            --border-dashed: #666;
             --grid-color-center: #666; /* Kolor linii centralnych siatki */
             --grid-color-lines: #444; /* Kolor pozostałych linii siatki */
        }

        html {
             box-sizing: border-box;
             height: 100%;
        }
        *, *:before, *:after {
             box-sizing: inherit;
        }

        body {
            margin: 0;
            padding: 0;
            background-color: var(--bg-main);
            color: var(--text-light);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol";
            display: flex;
            flex-direction: column;
            height: 100vh; /* Cała wysokość widoku */
            overflow: hidden; /* Zapobiega przewijaniu całej strony */
            touch-action: none; /* Optymalizacja dla dotyku na całym body */
        }

        .header {
            padding: 10px;
            text-align: center;
            background-color: var(--bg-header);
            flex-shrink: 0; /* Nagłówek nie będzie się kurczył */
            border-bottom: 1px solid var(--border-color);
        }

        .header h1 {
            color: var(--accent-color);
            text-shadow: 0 0 8px var(--accent-color), 0 0 15px var(--accent-color);
            margin: 0;
            /* Użyj clamp dla responsywnej czcionki z ograniczeniami */
            font-size: clamp(18px, 2.5vw, 28px);
        }

        .header p {
            color: var(--accent-color);
            text-shadow: 0 0 5px var(--accent-color);
            margin: 5px 0 0 0;
            font-size: clamp(12px, 1.2vw, 16px);
        }

        .container {
            display: flex;
            flex: 1; /* Rozciąga się, aby wypełnić dostępną przestrzeń */
            overflow: hidden; /* Zapobiega problemom z układem */
            flex-direction: row; /* Domyślny układ: eksplorator po lewej, viewer po prawej */
        }

        .explorer {
            width: 400px; /* Domyślna szerokość eksploratora */
            background-color: var(--bg-explorer);
            border-right: 1px solid var(--border-color);
            display: flex;
            flex-direction: column; /* Układ pionowy: lista plików na górze, panel na dole */
            overflow: hidden; /* Zapobiega przepełnieniu */
            transition: width 0.3s ease; /* Płynna zmiana szerokości */
            flex-shrink: 0; /* Zapobiega kurczeniu się */
        }

        .explorer-content {
            flex: 1; /* Główna część eksploratora rozciąga się */
            overflow-y: auto; /* Przewijanie dla samej listy plików */
            padding: 10px;
             overscroll-behavior: contain; /* Zapobiega przewijaniu body */
            font-size: 10px; /* Zmniejszona czcionka w eksploratorze */
        }

        .copy-panel {
            padding: 10px;
            background-color: var(--bg-copy-panel);
            border-top: 1px solid var(--border-dashed);
            flex-shrink: 0; /* Panel nie kurczy się */
        }

        .copy-input {
            display: flex;
        }

        .copy-input input[type="text"] {
            flex: 1; /* Pole tekstowe zajmuje dostępną szerokość */
            padding: 8px;
            border: none;
            border-radius: 4px 0 0 4px; /* Zaokrąglenie lewych rogów */
            background-color: #eee;
            color: var(--text-dark);
            font-size: 12px; /* Mniejsza czcionka dla linku */
            min-width: 0; /* Pozwala na kurczenie się w flex */
            border: 1px solid var(--border-color);
            border-right: none;
        }

        .copy-input button {
            padding: 8px 12px;
            background-color: var(--accent-color);
            color: var(--text-dark);
            border: 1px solid var(--accent-color);
            border-radius: 0 4px 4px 0; /* Zaokrąglenie prawych rogów */
            cursor: pointer;
            white-space: nowrap; /* Zapobiega łamaniu tekstu */
            font-weight: bold;
            transition: background-color 0.2s ease;
        }
        .copy-input button:hover {
             background-color: #6ee66e; /* Jaśniejszy zielony */
             border-color: #6ee66e;
        }

        .viewer {
            flex: 1; /* Viewer zajmuje resztę dostępnej przestrzeni */
            position: relative; /* Potrzebne dla pozycjonowania */
            background-color: var(--bg-viewer);
            min-width: 0; /* Zapobiega rozpychaniu przez zawartość */
             overflow: hidden; /* Upewnia się, że canvas nie wychodzi poza */
        }

        /* Style elementów drzewa plików (bez zmian) */
        .tree-item {
             margin-bottom: 2px; /* Mały odstęp */
        }

        .directory, .file {
            padding: 6px 8px; /* Mniejszy padding pionowy */
            cursor: pointer;
            border-radius: 4px;
            word-break: break-all; /* Łamanie długich nazw */
            user-select: none; /* Zapobiega zaznaczaniu tekstu */
            transition: background-color 0.2s ease;
            overflow: hidden; /* Ukrywa nadmiarowy tekst */
            text-overflow: ellipsis; /* Dodaje "..." dla długich nazw */
            white-space: nowrap; /* Zapobiega zawijaniu tekstu */
            font-size: 13px; /* Zmniejszona czcionka */
        }

        .directory {
            background-color: var(--bg-directory);
            color: var(--accent-color);
            font-weight: bold;
        }
        .directory:hover {
            background-color: var(--bg-directory-hover);
        }

        .file {
            background-color: var(--bg-file);
            color: var(--text-light);
        }
        .file.active { /* Styl dla aktywnie wybranego pliku */
            background-color: var(--bg-file-active);
            border-left: 3px solid var(--border-accent);
            padding-left: 5px; /* Korekta paddingu */
            font-weight: bold;
            color: var(--accent-color);
        }
        .file:hover {
            background-color: var(--bg-file-hover);
        }

        .tree-children {
            margin-left: 15px; /* Wcięcie dla zawartości katalogu */
            border-left: 1px dashed var(--border-dashed);
            padding-left: 10px; /* Odstęp od linii */
        }

        .toggle { /* Styl dla przełącznika +/- */
            display: inline-block;
            width: 15px;
            text-align: center;
            font-weight: bold;
            margin-right: 5px;
            color: var(--accent-color);
        }

        #model-container { /* Kontener dla renderera Three.js */
            width: 100%;
            height: 100%;
            display: block; /* Upewnia się, że zajmuje całą przestrzeń */
             /* touch-action: none; Zostało przeniesione na body */
        }
         /* Styl dla canvasu Three.js */
         #model-container canvas {
             display: block; /* Usuwa ewentualne dodatkowe marginesy */
         }

        /* --- Styl dla informacji o skali siatki --- */
        #grid-scale-info {
            position: absolute;
            bottom: 10px;
            right: 10px;
            padding: 5px 10px;
            background-color: rgba(0, 0, 0, 0.5); /* Półprzezroczyste tło */
            color: var(--text-light);
            font-size: 12px;
            border-radius: 3px;
            z-index: 10; /* Upewnij się, że jest nad canvasem */
            pointer-events: none; /* Nie przechwytuje kliknięć */
            visibility: hidden; /* Ukryte na początku */
        }

        /* === Media Queries dla urządzeń mobilnych (bez zmian) === */

        /* --- Układ pionowy (Portrait) --- */
        @media (max-width: 768px) and (orientation: portrait) {
            body {
                touch-action: manipulation;
            }
            .container {
                flex-direction: column;
            }
            .header h1 { font-size: 5.5vw; }
            .header p { font-size: 3vw; }

            .explorer {
                width: 100%;
                height: 35vh;
                max-height: 280px;
                border-right: none;
                border-bottom: 1px solid var(--border-color);
            }
             .explorer-content {
                 padding: 8px;
                 font-size: 12px;
             }
             .copy-panel { padding: 8px; }
             .viewer {
                 flex: 1;
             }
             #model-container {
                  touch-action: none;
             }
            #grid-scale-info {
                bottom: 5px;
                right: 5px;
                font-size: 10px; /* Mniejsza czcionka na mobilnych */
                padding: 3px 6px;
            }
        }

        /* --- Układ poziomy (Landscape) --- */
        @media (max-height: 500px) and (orientation: landscape) {
             body {
                 touch-action: manipulation;
             }
             .container {
                 flex-direction: row;
             }
             .header h1 {
                 font-size: clamp(10px, 2vw, 20px);
                 text-shadow: 0 0 5px var(--accent-color);
             }
             .header p {
                 font-size: clamp(10px, 1vw, 12px);
                 text-shadow: none;
                 margin-top: 2px;
             }
             .header {
                  padding: 5px 10px;
             }

             .explorer {
                 width: 43%;
                 max-width: 350px;
                 height: 100%;
                 max-height: none;
                 border-right: 1px solid var(--border-color);
                 border-bottom: none;
             }
              .explorer-content {
                  padding: 5px;
                  font-size: 6px;
              }

             .copy-panel {
                 padding: 5px;
             }
              .copy-input input[type="text"],
              .copy-input button {
                  padding: 5px 8px;
                  font-size: 6px;
              }

             .viewer {
                 flex: 1;
             }
              #model-container {
                  touch-action: none;
             }
             #grid-scale-info {
                font-size: 10px;
                padding: 3px 6px;
            }
        }

    </style>
</head>
<body oncontextmenu="return false;">
    <div class="header">
        <h1>Galeria 3D - Skandy 3D LiDAR *.glb</h1>
        <p>(Przytrzymaj i za pomocą rolki i ruchów myszy lub na smartfonie za pomocą palców przybliżaj, oddalaj oraz obracaj model 3D.)</p>
    </div>

    <div class="container">
        <div class="explorer">
            <div class="explorer-content">
                <?php echo renderTree($filesTree); /* Wyrenderowanie drzewa plików */ ?>
            </div>
            <div class="copy-panel">
                <div class="copy-input">
                    <input type="text" id="copy-url" value="<?php echo htmlspecialchars($initialShareUrl, ENT_QUOTES, 'UTF-8'); ?>" readonly title="Link do aktualnie wybranego modelu">
                    <button onclick="copyShareUrl()" title="Kopiuj link do schowka">Kopiuj</button>
                </div>
            </div>
        </div>

        <div class="viewer">
            <div id="model-container"></div>
            <div id="grid-scale-info">Skala siatki: -</div>
        </div>
    </div>

    <script>
        let scene, camera, renderer, model, controls, gridHelper;
        let isMobile = /Mobi|Android/i.test(navigator.userAgent);
        let currentModelPath = ''; // Przechowuje ścieżkę do aktualnie załadowanego modelu
        const baseShareUrl = '<?php echo $currentUrlBase; ?>'; // Podstawowy URL bez parametrów
        const modelContainer = document.getElementById('model-container'); // Cache'owanie elementu
        const gridScaleInfo = document.getElementById('grid-scale-info'); // Cache'owanie elementu informacji o siatce

        // Definicja kolorów siatki pobranych ze stylów CSS
        const gridColorCenter = new THREE.Color(getComputedStyle(document.documentElement).getPropertyValue('--grid-color-center').trim() || 0x666666);
        const gridColorLines = new THREE.Color(getComputedStyle(document.documentElement).getPropertyValue('--grid-color-lines').trim() || 0x444444);

        // Definicja progu rozmiaru dla zmiany siatki (w metrach)
        const SMALL_OBJECT_THRESHOLD = 0.5; // Obiekty mniejsze niż 0.5m (50cm) będą miały siatkę cm

        function initThreeJS() {
            // --- Inicjalizacja Sceny ---
            scene = new THREE.Scene();
            scene.background = new THREE.Color(getComputedStyle(document.documentElement).getPropertyValue('--bg-viewer').trim() || 0x222222);

            // --- Inicjalizacja Kamery ---
            camera = new THREE.PerspectiveCamera(50, modelContainer.clientWidth / modelContainer.clientHeight, 0.1, 1000);
            camera.position.set(0, 1.6, 5);
            camera.lookAt(0, 0, 0);

            // --- Inicjalizacja Renderera ---
            renderer = new THREE.WebGLRenderer({
                antialias: true,
                alpha: false,
                powerPreference: "high-performance"
            });
            renderer.setPixelRatio(window.devicePixelRatio);
            renderer.setSize(modelContainer.clientWidth, modelContainer.clientHeight);
            renderer.domElement.style.display = 'block';
            modelContainer.appendChild(renderer.domElement);

            // --- Kontrola Kamery (OrbitControls) ---
            controls = new THREE.OrbitControls(camera, renderer.domElement);
            controls.enableDamping = true;
            controls.dampingFactor = 0.1;
            controls.screenSpacePanning = false;
            controls.maxPolarAngle = Math.PI;
            controls.minPolarAngle = 0;
            controls.enablePan = true;
            controls.zoomSpeed = 0.8;
            controls.rotateSpeed = 0.5;

            // --- Oświetlenie ---
            const ambientLight = new THREE.AmbientLight(0xcccccc, 1.0);
            scene.add(ambientLight);
            const directionalLight1 = new THREE.DirectionalLight(0xffffff, 0.8);
            directionalLight1.position.set(5, 10, 7);
            scene.add(directionalLight1);
            const directionalLight2 = new THREE.DirectionalLight(0xffffff, 0.5);
            directionalLight2.position.set(-5, -5, -5);
            scene.add(directionalLight2);

            // --- Początkowe Podłoże z Siatką (zostanie zastąpione w centerModel) ---
            // Tworzymy domyślną siatkę 1m/1m, która zostanie usunięta i zastąpiona przy ładowaniu modelu
            createOrUpdateGrid(100, 100, "1m/1m", 0); // Duża siatka 100m, podziałka 1m
            updateGridScaleInfo("1m/1m"); // Pokaż domyślną skalę
            gridScaleInfo.style.visibility = 'visible'; // Pokaż informację o skali

            // --- Pętla Animacji ---
            function animate() {
                requestAnimationFrame(animate);
                if (modelContainer.offsetParent !== null) {
                    controls.update();
                    renderer.render(scene, camera);
                }
            }
            animate();

            // --- Obsługa Zmiany Rozmiaru Okna ---
            function onWindowResize() {
                const width = modelContainer.clientWidth;
                const height = modelContainer.clientHeight;
                if (width > 0 && height > 0) {
                    camera.aspect = width / height;
                    camera.updateProjectionMatrix();
                    renderer.setSize(width, height);
                }
            }
            const resizeObserver = new ResizeObserver(entries => {
                for (let entry of entries) {
                    if (entry.target === modelContainer) {
                        onWindowResize();
                    }
                }
            });
            resizeObserver.observe(modelContainer);

            // --- Ładowanie modelu z parametru URL (jeśli istnieje) ---
            handleUrlParameters();

        } // Koniec initThreeJS

        // --- Funkcja obsługi parametrów URL (bez zmian) ---
        function handleUrlParameters() {
            const urlParams = new URLSearchParams(window.location.search);
            const modelParam = urlParams.get('model');
            if (modelParam) {
                const decodedModelParam = decodeURIComponent(modelParam); // Dekodowanie ścieżki
                const fileElement = document.querySelector(`.file[data-path="${CSS.escape(decodedModelParam)}"]`); // Użyj CSS.escape dla bezpieczeństwa selektora
                if (fileElement) {
                    setTimeout(() => {
                        loadModel(decodedModelParam, fileElement);
                        let parent = fileElement.closest('.tree-children');
                        while (parent) {
                            if (parent.style.display === 'none') {
                                const dirElement = parent.previousElementSibling;
                                if (dirElement && dirElement.classList.contains('directory')) {
                                    toggleDirectory(dirElement, true);
                                }
                            }
                            parent = parent.parentElement.closest('.tree-children');
                        }
                        fileElement.scrollIntoView({ behavior: 'auto', block: 'center' });
                    }, 100);
                } else {
                    console.warn(`Model specified in URL not found in file tree: ${decodedModelParam}`);
                    updateCopyUrl(baseShareUrl);
                    clearScene(); // Wyczyść scenę, jeśli model z URL nie istnieje
                }
            } else {
                updateCopyUrl(baseShareUrl);
                clearScene(); // Wyczyść scenę przy braku parametru
            }
        }


        // --- Funkcja Ładowania Modelu GLB (zmodyfikowana) ---
        function loadModel(path, element) {
            if (!path) return;

            // Najpierw wyczyść poprzedni model i siatkę
            clearSceneContent();

            document.querySelectorAll('.file').forEach(el => el.classList.remove('active'));

            if (element) {
                element.classList.add('active');
            }

            currentModelPath = path;
            updateCopyUrl();

            modelContainer.style.cursor = 'wait';
            gridScaleInfo.style.visibility = 'hidden'; // Ukryj info podczas ładowania

            const loader = new THREE.GLTFLoader();
            loader.load(
                path,
                function(gltf) {
                    model = gltf.scene;
                    scene.add(model);
                    centerModel(); // Centrowanie i dostosowanie siatki
                    modelContainer.style.cursor = 'grab';
                    gridScaleInfo.style.visibility = 'visible'; // Pokaż info po załadowaniu
                },
                undefined,
                function(error) {
                    console.error('Error loading model:', error);
                    alert('Nie udało się załadować modelu: ' + path + '\nSzczegóły w konsoli.');
                    clearScene(); // Wyczyść scenę w razie błędu
                    modelContainer.style.cursor = 'default';
                    gridScaleInfo.style.visibility = 'visible'; // Pokaż domyślne info o siatce
                }
            );
        }

        // --- Funkcja pomocnicza do czyszczenia materiałów (bez zmian) ---
         const cleanMaterial = material => {
             material.dispose();
             for (const key of Object.keys(material)) {
                 const value = material[key];
                 if (value && typeof value === 'object' && 'minFilter' in value) {
                     value.dispose();
                 }
             }
         };

        // --- Funkcja usuwająca model i siatkę ze sceny ---
        function clearSceneContent() {
            if (model) {
                scene.remove(model);
                model.traverse(child => {
                     if (child.isMesh) {
                         child.geometry.dispose();
                         if (child.material.isMaterial) {
                             cleanMaterial(child.material);
                         } else {
                             for (const material of child.material) cleanMaterial(material);
                         }
                     }
                 });
                model = null;
            }
            if (gridHelper) {
                scene.remove(gridHelper);
                gridHelper.geometry.dispose();
                gridHelper.material.dispose();
                gridHelper = null;
            }
        }


        // --- Funkcja Tworzenia lub Aktualizacji Siatki ---
        function createOrUpdateGrid(size, divisions, scaleText, gridYPosition) {
             if (gridHelper) {
                scene.remove(gridHelper);
                gridHelper.geometry.dispose();
                gridHelper.material.dispose();
            }
            gridHelper = new THREE.GridHelper(size, divisions, gridColorCenter, gridColorLines);
            gridHelper.position.y = gridYPosition;
            scene.add(gridHelper);
            updateGridScaleInfo(scaleText);
        }

        // --- Funkcja Aktualizacji Tekstu Skali Siatki ---
        function updateGridScaleInfo(scaleText) {
            if (gridScaleInfo) {
                gridScaleInfo.textContent = `Skala siatki: ${scaleText}`;
            }
        }

        // --- Funkcja Centrowania Modelu (zmodyfikowana) ---
        function centerModel() {
            if (!model) return;

            const box = new THREE.Box3().setFromObject(model);
            if (box.isEmpty()) {
                 console.warn("Model bounding box is empty. Cannot center.");
                 controls.target.set(0, 0, 0);
                 // Stwórz domyślną dużą siatkę, jeśli nie można wycentrować
                 createOrUpdateGrid(100, 100, "1m/1m", 0);
                 controls.update();
                 return;
             }

            const center = box.getCenter(new THREE.Vector3());
            const size = box.getSize(new THREE.Vector3());

            // Wyśrodkuj model geometrycznie
            model.position.sub(center);

            // Ustaw pozycję Y siatki na dolnej krawędzi modelu
            const gridYPosition = box.min.y - center.y;

            // --- === ZMIANA: Dynamiczne tworzenie siatki === ---
            const maxDim = Math.max(size.x, size.y, size.z);
            let gridScaleText, gridSize, gridDivisions;

            if (maxDim < SMALL_OBJECT_THRESHOLD && maxDim > 0.001) { // Jeśli obiekt jest "mały" (np. < 50cm) i nie zerowy
                gridScaleText = "1cm/1cm";
                // Siatka 1m x 1m z podziałką co 1cm = 100 działek
                gridSize = 1;
                gridDivisions = 100;
            } else { // W przeciwnym razie obiekt jest "duży"
                gridScaleText = "1m/1m";
                 // Większa siatka, np. 100m x 100m z podziałką co 1m = 100 działek
                 // Można dostosować rozmiar, np. do 10 * maxDim, ale stały duży rozmiar jest prostszy
                 gridSize = 100; // Duża płaszczyzna 100m x 100m
                 gridDivisions = 100; // Podziałka co 1 metr
            }
            // Utwórz lub zaktualizuj siatkę z odpowiednimi parametrami
            createOrUpdateGrid(gridSize, gridDivisions, gridScaleText, gridYPosition);
            // --- === KONIEC ZMIANY === ---


            // Dostosuj kamerę (logika bez zmian)
             const effectiveDim = maxDim > 0.01 ? maxDim : Math.max(size.x, size.z) > 0.01 ? Math.max(size.x, size.z) : 1;
            const fov = camera.fov * (Math.PI / 180);
            let cameraDistance = Math.abs(effectiveDim / 2 / Math.tan(fov / 2));
            cameraDistance *= 1.7;
            cameraDistance = Math.max(0.5, cameraDistance);
             cameraDistance = Math.min(500, cameraDistance);

             const cameraY = Math.max(size.y / 2, effectiveDim * 0.3);
            camera.position.set(0, cameraY, cameraDistance);
            controls.target.set(0, 0, 0);

            controls.minDistance = cameraDistance / 5;
            controls.maxDistance = cameraDistance * 5;

            controls.update();
        }

        // --- Funkcja Rozwijania/Zwijania Katalogu (bez zmian) ---
        function toggleDirectory(element, forceOpen = false) {
            const parent = element.parentElement;
            const children = parent.querySelector(':scope > .tree-children');
            const toggle = element.querySelector(':scope > .toggle');

            if (!children || !toggle) return;

            const isHidden = children.style.display === 'none';

             if (forceOpen && isHidden) {
                 children.style.display = 'block';
                 toggle.textContent = '-';
             } else if (!forceOpen) {
                  if (isHidden) {
                     children.style.display = 'block';
                     toggle.textContent = '-';
                 } else {
                     children.style.display = 'none';
                     toggle.textContent = '+';
                 }
             }
        }


        // --- Funkcja Aktualizacji URL w Polu do Kopiowania (bez zmian) ---
        function updateCopyUrl(urlToSet = null) {
            const copyInput = document.getElementById('copy-url');
            if (!copyInput) return;

            let finalUrl;
            if (urlToSet) {
                finalUrl = urlToSet;
            } else if (currentModelPath) {
                const url = new URL(baseShareUrl);
                url.searchParams.set('model', encodeURIComponent(currentModelPath));
                finalUrl = url.toString();
            } else {
                finalUrl = baseShareUrl;
            }
            copyInput.value = finalUrl;

            try {
                history.replaceState({ path: currentModelPath }, '', finalUrl);
            } catch (e) {
                console.warn("Could not update browser history:", e);
            }
        }

        // --- Funkcja Kopiowania URL do Schowka (bez zmian) ---
        function copyShareUrl() {
            const copyInput = document.getElementById('copy-url');
            if (!copyInput || !copyInput.value) return;

            const copyButton = copyInput.nextElementSibling;

            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(copyInput.value).then(() => {
                    showCopyFeedback(copyButton, 'Skopiowano!');
                }).catch(err => {
                    console.error('Async clipboard write failed: ', err);
                    showCopyFeedback(copyButton, 'Błąd kopiowania');
                     fallbackCopyTextToClipboard(copyInput, copyButton);
                });
            } else {
                fallbackCopyTextToClipboard(copyInput, copyButton);
            }
        }

        // Zapasowa funkcja kopiowania (bez zmian)
        function fallbackCopyTextToClipboard(inputElement, buttonElement) {
            inputElement.select();
            inputElement.setSelectionRange(0, 99999);
            let success = false;
            try {
                success = document.execCommand('copy');
            } catch (err) {
                console.error('Fallback document.execCommand failed: ', err);
                success = false;
            }
            showCopyFeedback(buttonElement, success ? 'Skopiowano!' : 'Błąd kopiowania');
            if (!success) {
                 alert("Nie można skopiować automatycznie. Proszę skopiować link ręcznie.");
            }
             window.getSelection().removeAllRanges();
        }

        // Funkcja pomocnicza do pokazywania informacji zwrotnej na przycisku (bez zmian)
        function showCopyFeedback(button, message) {
             if (!button) return;
            const originalText = button.textContent;
             const originalTitle = button.title;
            button.textContent = message;
             button.title = message;
             button.style.backgroundColor = (message === 'Skopiowano!') ? '#a0ffa0' : '#ffa0a0';

            setTimeout(() => {
                button.textContent = originalText;
                 button.title = originalTitle;
                 button.style.backgroundColor = '';
            }, 2000);
        }


        // --- Inicjalizacja po Załadowaniu Strony ---
        window.onload = function() {
            initThreeJS();
        };

        // --- Blokada Prawego Przycisku Myszy (bez zmian) ---
        document.addEventListener('contextmenu', function(e) {
            if (e.target instanceof HTMLInputElement || e.target instanceof HTMLTextAreaElement) {
                return;
            }
            e.preventDefault();
            return false;
        });

        // --- Obsługa Nawigacji Wstecz/Dalej (zmodyfikowana) ---
         window.addEventListener('popstate', (event) => {
             const urlParams = new URLSearchParams(window.location.search);
             const modelParam = urlParams.get('model');
             const decodedModelParam = modelParam ? decodeURIComponent(modelParam) : null;

             if (decodedModelParam && decodedModelParam !== currentModelPath) {
                 const fileElement = document.querySelector(`.file[data-path="${CSS.escape(decodedModelParam)}"]`);
                 if (fileElement) {
                     loadModel(decodedModelParam, fileElement); // Ładuje model i ustawia poprawną siatkę
                     setTimeout(() => { // Otwiera drzewo
                         let parent = fileElement.closest('.tree-children');
                         while(parent) {
                              if (parent.style.display === 'none') {
                                 const dirElement = parent.previousElementSibling;
                                 if (dirElement && dirElement.classList.contains('directory')) {
                                      toggleDirectory(dirElement, true);
                                 }
                             }
                             parent = parent.parentElement.closest('.tree-children');
                         }
                         fileElement.scrollIntoView({ behavior: 'auto', block: 'center' });
                     }, 100);
                 } else {
                     console.warn("Model from history popstate not found:", decodedModelParam);
                     clearScene(); // Czyści scenę, jeśli model nie został znaleziony
                 }
             } else if (!decodedModelParam && currentModelPath) {
                 // URL nie ma modelu, a jakiś jest załadowany - wyczyść scenę
                 clearScene();
             }
             // Aktualizacja URL w polu kopiowania nie jest potrzebna tutaj,
             // bo loadModel i clearScene już to robią.
         });

         // Funkcja do czyszczenia sceny (model, siatka, stan)
         function clearScene() {
              clearSceneContent(); // Usuwa model i siatkę
              currentModelPath = '';
              document.querySelectorAll('.file').forEach(el => el.classList.remove('active'));
              // Przywróć domyślną siatkę i info
              createOrUpdateGrid(100, 100, "1m/1m", 0);
              gridScaleInfo.style.visibility = 'visible';
              // Opcjonalnie resetuj kamerę/kontroler
              camera.position.set(0, 1.6, 5);
              controls.target.set(0, 0, 0);
              controls.minDistance = 0.1; // Reset zoom limits
              controls.maxDistance = Infinity;
              controls.update();
              updateCopyUrl(baseShareUrl); // Pokazuje bazowy URL
         }

    </script>
</body>
</html>